/*
 * Copyright © 2013 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 * Authors:
 *    Brad Volkin <bradley.d.volkin@intel.com>
 *
 */

#include "i915_drv.h"

#define STD_MI_OPCODE_MASK  0xFF800000
#define STD_3D_OPCODE_MASK  0xFFFF0000
#define STD_2D_OPCODE_MASK  0xFFC00000
#define STD_MFX_OPCODE_MASK 0xFFFF0000

#define CMD(op, opm, f, lm, fl, ...)		\
	{					\
		.flags = (fl) | (f),		\
		.cmd = { (op), (opm) },		\
		.length = { (lm) },		\
		__VA_ARGS__			\
	}

/* Convenience macros to compress the tables */
#define SMI STD_MI_OPCODE_MASK
#define S3D STD_3D_OPCODE_MASK
#define S2D STD_2D_OPCODE_MASK
#define SMFX STD_MFX_OPCODE_MASK
#define F CMD_DESC_FIXED
#define S CMD_DESC_SKIP
#define R CMD_DESC_REJECT
#define W CMD_DESC_REGISTER
#define B CMD_DESC_BITMASK

/*            Command                          Mask   Fixed Len   Action
	      ---------------------------------------------------------- */
static const struct drm_i915_cmd_descriptor common_cmds[] = {
	CMD(  MI_NOOP,                          SMI,    F,  1,      S  ),
	CMD(  MI_USER_INTERRUPT,                SMI,    F,  1,      R  ),
	CMD(  MI_WAIT_FOR_EVENT,                SMI,    F,  1,      S  ),
	CMD(  MI_ARB_CHECK,                     SMI,    F,  1,      S  ),
	CMD(  MI_REPORT_HEAD,                   SMI,    F,  1,      S  ),
	CMD(  MI_SUSPEND_FLUSH,                 SMI,    F,  1,      S  ),
	CMD(  MI_SEMAPHORE_MBOX,                SMI,   !F,  0xFF,   R  ),
	CMD(  MI_STORE_DWORD_INDEX,             SMI,   !F,  0xFF,   R  ),
	CMD(  MI_LOAD_REGISTER_IMM(1),          SMI,   !F,  0xFF,   W,
		.reg = { .offset = 1, .mask = 0x007FFFFC }             ),
	CMD(  MI_UPDATE_GTT,                    SMI,   !F,  0xFF,   R  ),
	CMD(  MI_STORE_REGISTER_MEM(1),         SMI,   !F,  0xFF,   W | B,
		.reg = { .offset = 1, .mask = 0x007FFFFC },
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_LOAD_REGISTER_MEM,             SMI,   !F,  0xFF,   W | B,
		.reg = { .offset = 1, .mask = 0x007FFFFC },
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_BATCH_BUFFER_START,            SMI,   !F,  0xFF,   S  ),
};

static const struct drm_i915_cmd_descriptor render_cmds[] = {
	CMD(  MI_FLUSH,                         SMI,    F,  1,      S  ),
	CMD(  MI_ARB_ON_OFF,                    SMI,    F,  1,      R  ),
	CMD(  MI_DISPLAY_FLIP,                  SMI,   !F,  0xFF,   R  ),
	CMD(  MI_PREDICATE,                     SMI,    F,  1,      S  ),
	CMD(  MI_TOPOLOGY_FILTER,               SMI,    F,  1,      S  ),
	CMD(  MI_STORE_DWORD_IMM,               SMI,   !F,  0x3F,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_CLFLUSH,                       SMI,   !F,  0x3FF,  B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_REPORT_PERF_COUNT,             SMI,   !F,  0x3F,   B,
		.bits = {{
			.offset = 1,
			.mask = MI_REPORT_PERF_COUNT_GGTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_CONDITIONAL_BATCH_BUFFER_END,  SMI,   !F,  0xFF,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  GFX_OP_3DSTATE_VF_STATISTICS,     S3D,    F,  1,      S  ),
	CMD(  PIPELINE_SELECT,                  S3D,    F,  1,      S  ),
	CMD(  MEDIA_VFE_STATE,                  S3D,   !F,  0xFFFF, B,
		.bits = {{
			.offset = 2,
			.mask = MEDIA_VFE_STATE_MMIO_ACCESS_MASK,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  GPGPU_OBJECT,                     S3D,   !F,  0xFF,   S  ),
	CMD(  GPGPU_WALKER,                     S3D,   !F,  0xFF,   S  ),
	CMD(  GFX_OP_3DSTATE_SO_DECL_LIST,      S3D,   !F,  0x1FF,  S  ),
	CMD(  GFX_OP_PIPE_CONTROL(5),           S3D,   !F,  0xFF,   B,
		.bits = {{
			.offset = 1,
			.mask = (PIPE_CONTROL_MMIO_WRITE | PIPE_CONTROL_NOTIFY),
			.expected = 0
		},
		{
			.offset = 1,
			.mask = (PIPE_CONTROL_GLOBAL_GTT_IVB |
				 PIPE_CONTROL_STORE_DATA_INDEX),
			.expected = 0,
			.condition_offset = 1,
			.condition_mask = PIPE_CONTROL_POST_SYNC_OP_MASK
		}},
		.bits_count = 2                                        )
};

static const struct drm_i915_cmd_descriptor hsw_render_cmds[] = {
	CMD(  MI_SET_PREDICATE,                 SMI,    F,  1,      S  ),
	CMD(  MI_RS_CONTROL,                    SMI,    F,  1,      S  ),
	CMD(  MI_URB_ATOMIC_ALLOC,              SMI,    F,  1,      S  ),
	CMD(  MI_RS_CONTEXT,                    SMI,    F,  1,      S  ),
	CMD(  MI_LOAD_SCAN_LINES_INCL,          SMI,   !F,  0x3F,   R  ),
	CMD(  MI_LOAD_SCAN_LINES_EXCL,          SMI,   !F,  0x3F,   R  ),
	CMD(  MI_MATH,                          SMI,   !F,  0x3F,   S  ),
	CMD(  MI_LOAD_REGISTER_REG,             SMI,   !F,  0xFF,   W,
		.reg = { .offset = 1, .mask = 0x007FFFFC }             ),
	CMD(  MI_LOAD_URB_MEM,                  SMI,   !F,  0xFF,   S  ),
	CMD(  MI_STORE_URB_MEM,                 SMI,   !F,  0xFF,   S  ),
	CMD(  GFX_OP_3DSTATE_DX9_CONSTANTF_VS,  S3D,   !F,  0x7FF,  S  ),
	CMD(  GFX_OP_3DSTATE_DX9_CONSTANTF_PS,  S3D,   !F,  0x7FF,  S  ),
};

static const struct drm_i915_cmd_descriptor video_cmds[] = {
	CMD(  MI_ARB_ON_OFF,                    SMI,    F,  1,      R  ),
	CMD(  MI_STORE_DWORD_IMM,               SMI,   !F,  0xFF,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_FLUSH_DW,                      SMI,   !F,  0x3F,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_FLUSH_DW_NOTIFY,
			.expected = 0
		},
		{
			.offset = 1,
			.mask = MI_FLUSH_DW_USE_GTT,
			.expected = 0,
			.condition_offset = 0,
			.condition_mask = MI_FLUSH_DW_OP_MASK
		},
		{
			.offset = 0,
			.mask = MI_FLUSH_DW_STORE_INDEX,
			.expected = 0,
			.condition_offset = 0,
			.condition_mask = MI_FLUSH_DW_OP_MASK
		}},
		.bits_count = 3                                        ),
	CMD(  MI_CONDITIONAL_BATCH_BUFFER_END,  SMI,   !F,  0xFF,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	/* MFX_WAIT doesn't fit the way we handle length for most commands.
	 * It has a length field but it uses a non-standard length bias.
	 * It is always 1 dword though, so just treat it as fixed length.
	 */
	CMD(  MFX_WAIT,                         SMFX,   F,    1,      S  ),
};

static const struct drm_i915_cmd_descriptor vecs_cmds[] = {
	CMD(  MI_ARB_ON_OFF,                    SMI,    F,  1,      R  ),
	CMD(  MI_STORE_DWORD_IMM,               SMI,   !F,  0xFF,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_FLUSH_DW,                      SMI,   !F,  0x3F,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_FLUSH_DW_NOTIFY,
			.expected = 0
		},
		{
			.offset = 1,
			.mask = MI_FLUSH_DW_USE_GTT,
			.expected = 0,
			.condition_offset = 0,
			.condition_mask = MI_FLUSH_DW_OP_MASK
		},
		{
			.offset = 0,
			.mask = MI_FLUSH_DW_STORE_INDEX,
			.expected = 0,
			.condition_offset = 0,
			.condition_mask = MI_FLUSH_DW_OP_MASK
		}},		
		.bits_count = 3                                        ),
	CMD(  MI_CONDITIONAL_BATCH_BUFFER_END,  SMI,    !F,  0xFF,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        )
};

static const struct drm_i915_cmd_descriptor blt_cmds[] = {
	CMD(  MI_DISPLAY_FLIP,                  SMI,   !F,  0xFF,   R  ),
	CMD(  MI_STORE_DWORD_IMM,               SMI,   !F,  0x1FF,  B,
		.bits = {{
			.offset = 0,
			.mask = MI_GLOBAL_GTT,
			.expected = 0
		}},
		.bits_count = 1                                        ),
	CMD(  MI_FLUSH_DW,                      SMI,   !F,  0x3F,   B,
		.bits = {{
			.offset = 0,
			.mask = MI_FLUSH_DW_NOTIFY,
			.expected = 0
		},
		{
			.offset = 1,
			.mask = MI_FLUSH_DW_USE_GTT,
			.expected = 0,
			.condition_offset = 0,
			.condition_mask = MI_FLUSH_DW_OP_MASK
		},
		{
			.offset = 0,
			.mask = MI_FLUSH_DW_STORE_INDEX,
			.expected = 0,
			.condition_offset = 0,
			.condition_mask = MI_FLUSH_DW_OP_MASK
		}},
		.bits_count = 3                                        ),
	CMD(  COLOR_BLT,                        S2D,   !F,  0x3F,   S  ),
	CMD(  SRC_COPY_BLT,                     S2D,   !F,  0x3F,   S  ),
};

static const struct drm_i915_cmd_descriptor hsw_blt_cmds[] = {
	CMD(  MI_LOAD_SCAN_LINES_INCL,          SMI,   !F,  0x3F,   R  ),
	CMD(  MI_LOAD_SCAN_LINES_EXCL,          SMI,   !F,  0x3F,   R  )
};

#undef CMD
#undef SMI
#undef S3D
#undef S2D
#undef SMFX
#undef F
#undef S
#undef R
#undef W
#undef B

static const struct drm_i915_cmd_table gen7_render_cmds[] = {
	{ common_cmds, ARRAY_SIZE(common_cmds) },
	{ render_cmds, ARRAY_SIZE(render_cmds) },
};

static const struct drm_i915_cmd_table hsw_render_ring_cmds[] = {
	{ common_cmds, ARRAY_SIZE(common_cmds) },
	{ render_cmds, ARRAY_SIZE(render_cmds) },
	{ hsw_render_cmds, ARRAY_SIZE(hsw_render_cmds) },
};

static const struct drm_i915_cmd_table gen7_video_cmds[] = {
	{ common_cmds, ARRAY_SIZE(common_cmds) },
	{ video_cmds, ARRAY_SIZE(video_cmds) },
};

static const struct drm_i915_cmd_table hsw_vebox_cmds[] = {
	{ common_cmds, ARRAY_SIZE(common_cmds) },
	{ vecs_cmds, ARRAY_SIZE(vecs_cmds) },
};

static const struct drm_i915_cmd_table gen7_blt_cmds[] = {
	{ common_cmds, ARRAY_SIZE(common_cmds) },
	{ blt_cmds, ARRAY_SIZE(blt_cmds) },
};

static const struct drm_i915_cmd_table hsw_blt_ring_cmds[] = {
	{ common_cmds, ARRAY_SIZE(common_cmds) },
	{ blt_cmds, ARRAY_SIZE(blt_cmds) },
	{ hsw_blt_cmds, ARRAY_SIZE(hsw_blt_cmds) },
};

/* Register whitelists, sorted by increasing register offset.
 *
 * Some registers that userspace accesses are 64 bits. The register
 * access commands only allow 32-bit accesses. Hence, we have to include
 * entries for both halves of the 64-bit registers.
 */

static const u32 gen7_render_regs[] = {
	HS_INVOCATION_COUNT,
	HS_INVOCATION_COUNT + sizeof(u32),
	DS_INVOCATION_COUNT,
	DS_INVOCATION_COUNT + sizeof(u32),
	IA_VERTICES_COUNT,
	IA_VERTICES_COUNT + sizeof(u32),
	IA_PRIMITIVES_COUNT,
	IA_PRIMITIVES_COUNT + sizeof(u32),
	VS_INVOCATION_COUNT,
	VS_INVOCATION_COUNT + sizeof(u32),
	GS_INVOCATION_COUNT,
	GS_INVOCATION_COUNT + sizeof(u32),
	GS_PRIMITIVES_COUNT,
	GS_PRIMITIVES_COUNT + sizeof(u32),
	CL_INVOCATION_COUNT,
	CL_INVOCATION_COUNT + sizeof(u32),
	CL_PRIMITIVES_COUNT,
	CL_PRIMITIVES_COUNT + sizeof(u32),
	PS_INVOCATION_COUNT,
	PS_INVOCATION_COUNT + sizeof(u32),
	PS_DEPTH_COUNT,
	PS_DEPTH_COUNT + sizeof(u32),
	GEN7_SO_NUM_PRIMS_WRITTEN(0),
	GEN7_SO_NUM_PRIMS_WRITTEN(0) + sizeof(u32),
	GEN7_SO_NUM_PRIMS_WRITTEN(1),
	GEN7_SO_NUM_PRIMS_WRITTEN(1) + sizeof(u32),
	GEN7_SO_NUM_PRIMS_WRITTEN(2),
	GEN7_SO_NUM_PRIMS_WRITTEN(2) + sizeof(u32),
	GEN7_SO_NUM_PRIMS_WRITTEN(3),
	GEN7_SO_NUM_PRIMS_WRITTEN(3) + sizeof(u32),
	GEN7_SO_WRITE_OFFSET(0),
	GEN7_SO_WRITE_OFFSET(1),
	GEN7_SO_WRITE_OFFSET(2),
	GEN7_SO_WRITE_OFFSET(3),
	L3_CNTL_REG2_ADDRESS_OFFSET,
	L3_CNTL_REG3_ADDRESS_OFFSET,
	L3_SQC_REG1_ADDRESS_OFFSET,
	L3_TLB_REG_ADDRESS_OFFSET,
	GEN7_L3SQCREG4,
	OCL_MMIO_REG1
};

static const u32 gen7_blt_regs[] = {
	BCS_SWCTRL
};

static const u32 gen7_vcs_regs[] = {
	MFD_ERROR_STATUS,
	MFC_BITSTREAM_BYTECOUNT_FRAME,
	MFC_BITSTREAM_SE_BITCOUNT_FRAME,
	MFC_IMAGE_STATUS_MASK,
	MFC_IMAGE_STATUS_CONTROL,
	MFC_BITSTREAM_BYTECOUNT_SLICE
};

#define CLIENT_MASK      0xE0000000
#define SUBCLIENT_MASK   0x18000000
#define MI_CLIENT        0x00000000
#define RC_CLIENT        0x60000000
#define BC_CLIENT        0x40000000
#define MEDIA_SUBCLIENT  0x10000000

static u32 gen7_render_get_cmd_length_mask(u32 cmd_header)
{
	u32 client = cmd_header & CLIENT_MASK;
	u32 subclient = cmd_header & SUBCLIENT_MASK;

	if (client == MI_CLIENT)
		return 0x3F;
	else if (client == RC_CLIENT) {
		if (subclient == MEDIA_SUBCLIENT)
			return 0xFFFF;
		else
			return 0x00FF;
	}

	DRM_DEBUG_DRIVER("CMD: Abnormal rcs cmd length! 0x%08X\n", cmd_header);
	return 0;
}

static u32 gen7_bsd_get_cmd_length_mask(u32 cmd_header)
{
	u32 client = cmd_header & CLIENT_MASK;
	u32 subclient = cmd_header & SUBCLIENT_MASK;

	if (client == MI_CLIENT)
		return 0x3F;
	else if (client == RC_CLIENT) {
		if (subclient == MEDIA_SUBCLIENT)
			return 0x0FFF;
		else
			return 0x00FF;
	}

	DRM_DEBUG_DRIVER("CMD: Abnormal bsd cmd length! 0x%08X\n", cmd_header);
	return 0;
}

static u32 gen7_blt_get_cmd_length_mask(u32 cmd_header)
{
	u32 client = cmd_header & CLIENT_MASK;

	if (client == MI_CLIENT)
		return 0x3F;
	else if (client == BC_CLIENT)
		return 0xFF;

	DRM_DEBUG_DRIVER("CMD: Abnormal blt cmd length! 0x%08X\n", cmd_header);
	return 0;
}

void i915_cmd_parser_init_ring(struct intel_ring_buffer *ring)
{
	if (!IS_HASWELL(ring->dev))
		return;

	switch (ring->id) {
	case RCS:
		if (IS_HASWELL(ring->dev)) {
			ring->cmd_tables = hsw_render_ring_cmds;
			ring->cmd_table_count =
				ARRAY_SIZE(hsw_render_ring_cmds);
		} else {
			ring->cmd_tables = gen7_render_cmds;
			ring->cmd_table_count = ARRAY_SIZE(gen7_render_cmds);
		}

		ring->reg_table = gen7_render_regs;
		ring->reg_count = ARRAY_SIZE(gen7_render_regs);

		ring->get_cmd_length_mask = gen7_render_get_cmd_length_mask;
		break;
	case VCS:
		ring->cmd_tables = gen7_video_cmds;
		ring->cmd_table_count = ARRAY_SIZE(gen7_video_cmds);
		ring->get_cmd_length_mask = gen7_bsd_get_cmd_length_mask;
		ring->reg_table = gen7_vcs_regs;
		ring->reg_count = ARRAY_SIZE(gen7_vcs_regs);
		break;
	case BCS:
		if (IS_HASWELL(ring->dev)) {
			ring->cmd_tables = hsw_blt_ring_cmds;
			ring->cmd_table_count = ARRAY_SIZE(hsw_blt_ring_cmds);
		} else {
		ring->cmd_tables = gen7_blt_cmds;
		ring->cmd_table_count = ARRAY_SIZE(gen7_blt_cmds);
		}

		ring->reg_table = gen7_blt_regs;
		ring->reg_count = ARRAY_SIZE(gen7_blt_regs);

		ring->get_cmd_length_mask = gen7_blt_get_cmd_length_mask;
		break;
	case VECS:
		ring->cmd_tables = hsw_vebox_cmds;
		ring->cmd_table_count = ARRAY_SIZE(hsw_vebox_cmds);
		/* VECS can use the same length_mask function as VCS */
		ring->get_cmd_length_mask = gen7_bsd_get_cmd_length_mask;
		break;
	default:
		DRM_DEBUG("CMD: cmd_parser_init with unknown ring: %d\n",
			  ring->id);
		break;
	}
}

static const struct drm_i915_cmd_descriptor*
find_cmd_in_table(const struct drm_i915_cmd_table *table,
		  u32 cmd_header)
{
	int i;

	for (i = 0; i < table->count; i++) {
		const struct drm_i915_cmd_descriptor *desc = &table->table[i];
		u32 masked_cmd = desc->cmd.mask & cmd_header;
		u32 masked_value = desc->cmd.value & desc->cmd.mask;

		if (masked_cmd == masked_value)
			return desc;
	}

	return NULL;
}

/* Returns a pointer to a descriptor for the command specified by cmd_header.
 *
 * The caller must supply space for a default descriptor via the default_desc
 * parameter. If no descriptor for the specified command exists in the ring's
 * command parser tables, this function fills in default_desc based on the
 * ring's default length encoding and returns default_desc.
 */
static const struct drm_i915_cmd_descriptor*
find_cmd(struct intel_ring_buffer *ring,
	 u32 cmd_header,
	 struct drm_i915_cmd_descriptor *default_desc)
{
	u32 mask;
	int i;

	for (i = 0; i < ring->cmd_table_count; i++) {
		const struct drm_i915_cmd_descriptor *desc;

		desc = find_cmd_in_table(&ring->cmd_tables[i], cmd_header);
		if (desc)
			return desc;
	}

	mask = ring->get_cmd_length_mask(cmd_header);
	if (!mask)
		return NULL;

	BUG_ON(!default_desc);
	default_desc->flags = CMD_DESC_SKIP;
	default_desc->length.mask = mask;

	return default_desc;
}

static int valid_reg(const u32 *table, int count, u32 addr)
{
	if (table && count != 0) {
		int i;

		for (i = 0; i < count; i++) {
			if (table[i] == addr)
				return 1;
		}
	}

	return 0;
}

static u32 *vmap_batch(struct drm_i915_gem_object *obj)
{
	int i;
	void *addr = NULL;
	struct scatterlist *sg;
	struct page **pages;

	pages = drm_malloc_ab(obj->pages->nents, sizeof(*pages));
	if (pages == NULL) {
		DRM_DEBUG("Failed to get space for pages\n");
		return NULL;
	}

	for_each_sg(obj->pages->sgl, sg, obj->pages->nents, i)
		pages[i] = sg_page(sg);

	addr = vmap(pages, obj->pages->nents, 0, PAGE_KERNEL);
	if (addr == NULL)
		DRM_DEBUG("Failed to vmap pages\n");

	drm_free_large(pages);
	return (u32*)addr;
}

int i915_needs_cmd_parser(struct intel_ring_buffer *ring)
{
	drm_i915_private_t *dev_priv =
		(drm_i915_private_t *)ring->dev->dev_private;

	/* No command tables indicates a platform without parsing */
	if (!ring->cmd_tables)
		return 0;

	/* XXX: VLV is Gen7 and therefore has cmd_tables, but has PPGTT
	 * disabled. That will cause all of the parser's PPGTT checks to
	 * fail. For now, disable parsing when PPGTT is off.
	 */
	if (!dev_priv->mm.aliasing_ppgtt)
		return 0;

	return i915_enable_cmd_parser;
}

#define LENGTH_BIAS 2

int i915_parse_cmds(struct intel_ring_buffer *ring,
		    struct drm_i915_gem_object *batch_obj,
		    u32 batch_start_offset)
{
	int ret = 0;
	u32 *cmd, *batch_base, *batch_end;
	struct drm_i915_cmd_descriptor default_desc = { 0 };

	batch_base = vmap_batch(batch_obj);
	if (!batch_base) {
		DRM_DEBUG_DRIVER("CMD: Failed to vmap batch\n");
		return -ENOMEM;
	}

	cmd = batch_base + (batch_start_offset / sizeof(*cmd));
	batch_end = cmd + (batch_obj->base.size / sizeof(*batch_end));

	while (cmd < batch_end) {
		const struct drm_i915_cmd_descriptor *desc;
		u32 length;

		if (*cmd == MI_BATCH_BUFFER_END)
			break;

		desc = find_cmd(ring, *cmd, &default_desc);
		if (!desc) {
			DRM_DEBUG_DRIVER("CMD: Unrecognized command: 0x%08X\n",
					 *cmd);
			ret = -EINVAL;
			break;
		}

		if (desc->flags & CMD_DESC_FIXED)
			length = desc->length.fixed;
		else
			length = ((*cmd & desc->length.mask) + LENGTH_BIAS);

		if ((batch_end - cmd) < length) {
			DRM_DEBUG_DRIVER("CMD: Command length exceeds batch length: 0x%08X length=%d batchlen=%ld\n",
					 *cmd, length, batch_end - cmd);
			ret = -EINVAL;
			break;
		}

		if (desc->flags & CMD_DESC_REJECT) {
			DRM_DEBUG_DRIVER("CMD: Rejected command: 0x%08X\n", *cmd);
			ret = -EINVAL;
			break;
		}

		if (desc->flags & CMD_DESC_REGISTER) {
			u32 reg_addr = cmd[desc->reg.offset] & desc->reg.mask;

			if (!valid_reg(ring->reg_table,
				       ring->reg_count, reg_addr)) {
				DRM_DEBUG_DRIVER("CMD: Rejected register 0x%08X in command: 0x%08X (ring=%d)\n",
						 reg_addr, *cmd, ring->id);
				ret = -EINVAL;
				break;
			}
		}

		if (desc->flags & CMD_DESC_BITMASK) {
			int i;

			for (i = 0; i < desc->bits_count; i++) {
				u32 dword = cmd[desc->bits[i].offset] &
						desc->bits[i].mask;

				if (desc->bits[i].condition_mask != 0) {
					u32 offset =
						desc->bits[i].condition_offset;
					u32 condition = cmd[offset] &
						desc->bits[i].condition_mask;

					if (condition == 0)
						continue;
				}

				if (dword != desc->bits[i].expected) {
					DRM_DEBUG_DRIVER("CMD: Rejected command 0x%08X for bitmask 0x%08X (exp=0x%08X act=0x%08X) (ring=%d)\n",
							 *cmd, desc->bits[i].mask, desc->bits[i].expected, dword, ring->id);
					ret = -EINVAL;
					break;
				}
			}

			if (ret)
				break;
		}

		cmd +=length;
	}

	if (cmd >= batch_end) {
		DRM_DEBUG_DRIVER("CMD: Got to the end of the buffer w/o a BBE cmd!\n");
		ret = -EINVAL;
	}

	vunmap(batch_base);

	return ret;
}

