/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 */

#include "drmP.h"
#include "i915_drm.h"
#include "i915_drv.h"
#include "i915_trace.h"
#include "intel_drv.h"
#include <linux/swap.h>

static struct i915_gem_userptr_object *to_userptr_object(struct drm_i915_gem_object *obj)
{
	return container_of(obj, struct i915_gem_userptr_object, gem);
}

static int
i915_gem_userptr_get_pages(struct drm_i915_gem_object *obj)
{
	struct i915_gem_userptr_object *vmap = to_userptr_object(obj);
	int num_pages = obj->base.size >> PAGE_SHIFT;
	struct sg_table *st;
	struct scatterlist *sg;
	struct page **pvec;
	int n, pinned, ret;

	if (vmap->mm == NULL)
		return -EFAULT;

	if (!access_ok(vmap->read_only ? VERIFY_READ : VERIFY_WRITE,
		       (char __user *)vmap->user_ptr, vmap->user_size))
		return -EFAULT;

	/* If userspace should engineer that these pages are replaced in
	 * the vma between us binding this page into the GTT and completion
	 * of rendering... Their loss. If they change the mapping of their
	 * pages they need to create a new bo to point to the new vma.
	 *
	 * However, that still leaves open the possibility of the vma
	 * being copied upon fork. Which falls under the same userspace
	 * synchronisation issue as a regular bo, except that this time
	 * the process may not be expecting that a particular piece of
	 * memory is tied to the GPU.
	 *
	 */

	pvec = kmalloc(num_pages*sizeof(struct page *),
		       GFP_KERNEL | __GFP_NOWARN | __GFP_NORETRY);
	if (pvec == NULL) {
		pvec = drm_malloc_ab(num_pages, sizeof(struct page *));
		if (pvec == NULL)
			return -ENOMEM;
	}

	pinned = 0;
	if (vmap->mm == current->mm)
		pinned = __get_user_pages_fast(vmap->user_ptr, num_pages,
					       !vmap->read_only, pvec);
	if (pinned < num_pages) {
		struct mm_struct *mm = vmap->mm;
		ret = 0;
		mutex_unlock(&obj->base.dev->struct_mutex);
		down_read(&mm->mmap_sem);
		if (vmap->mm != NULL)
			ret = get_user_pages(current, mm,
					     vmap->user_ptr + (pinned << PAGE_SHIFT),
					     num_pages - pinned,
					     !vmap->read_only, 0,
					     pvec + pinned,
					     NULL);
		up_read(&mm->mmap_sem);
		mutex_lock(&obj->base.dev->struct_mutex);
		if (ret > 0)
			pinned += ret;

		if (obj->pages || pinned < num_pages) {
			ret = obj->pages ? 0 : -EFAULT;
			goto cleanup_pinned;
		}
	}

	st = kmalloc(sizeof(*st), GFP_KERNEL);
	if (st == NULL) {
		ret = -ENOMEM;
		goto cleanup_pinned;
	}

	if (sg_alloc_table(st, num_pages, GFP_KERNEL)) {
		ret = -ENOMEM;
		goto cleanup_st;
	}

	for_each_sg(st->sgl, sg, num_pages, n)
		sg_set_page(sg, pvec[n], PAGE_SIZE, 0);
	drm_free_large(pvec);

	obj->pages = st;
	return 0;

cleanup_st:
	kfree(st);
cleanup_pinned:
	release_pages(pvec, pinned, 0);
	drm_free_large(pvec);
	return ret;
}

static void
i915_gem_userptr_put_pages(struct drm_i915_gem_object *obj)
{
	struct scatterlist *sg;
	int i;

	if (obj->madv != I915_MADV_WILLNEED)
		obj->dirty = 0;

	for_each_sg(obj->pages->sgl, sg, obj->pages->nents, i) {
		struct page *page = sg_page(sg);

		if (obj->dirty)
			set_page_dirty(page);

		mark_page_accessed(page);
		page_cache_release(page);
	}
	obj->dirty = 0;

	sg_free_table(obj->pages);
	kfree(obj->pages);
}

static const struct drm_i915_gem_object_ops i915_gem_userptr_ops = {
	.get_pages = i915_gem_userptr_get_pages,
	.put_pages = i915_gem_userptr_put_pages,
};

/**
 * Creates a new mm object that wraps some user memory.
 */
int
i915_gem_userptr_ioctl(struct drm_device *dev, void *data, struct drm_file *file)
{
	struct drm_i915_private *dev_priv = dev->dev_private;
	struct drm_i915_gem_userptr *args = data;
	struct i915_gem_userptr_object *obj;
	loff_t first_data_page, last_data_page;
	int num_pages;
	int ret;
	u32 handle;

	if (args->flags & ~(I915_USERPTR_READ_ONLY | I915_USERPTR_UNSYNCHRONIZED))
		return -EINVAL;

	first_data_page = args->user_ptr / PAGE_SIZE;
	last_data_page = (args->user_ptr + args->user_size - 1) / PAGE_SIZE;
	num_pages = last_data_page - first_data_page + 1;
	if (num_pages * PAGE_SIZE > dev_priv->mm.gtt_total)
		return -E2BIG;

	/* Allocate the new object */
	obj = i915_gem_object_alloc(dev);
	if (obj == NULL)
		return -ENOMEM;

	if (drm_gem_private_object_init(dev, &obj->gem.base,
					num_pages * PAGE_SIZE)) {
		i915_gem_object_free(&obj->gem);
		return -ENOMEM;
	}

	i915_gem_object_init(&obj->gem, &i915_gem_userptr_ops);
	obj->gem.cache_level = I915_CACHE_LLC_MLC;

	obj->gem.gtt_offset = offset_in_page(args->user_ptr);
	obj->user_ptr = args->user_ptr;
	obj->user_size = args->user_size;
	obj->read_only = args->flags & I915_USERPTR_READ_ONLY;

	obj->mm = current->mm;

	ret = drm_gem_handle_create(file, &obj->gem.base, &handle);
	/* drop reference from allocate - handle holds it now */
	drm_gem_object_unreference(&obj->gem.base);
	if (ret)
		return ret;

	args->handle = handle;
	return 0;
}
