package com.wowza.wms.plugin.closedcaption.live;

import java.util.*;

import com.wowza.util.FLVUtils;
import com.wowza.util.StringUtils;
import com.wowza.wms.amf.*;
import com.wowza.wms.application.*;
import com.wowza.wms.client.IClient;
import com.wowza.wms.media.h264.*;
import com.wowza.wms.module.ModuleBase;
import com.wowza.wms.request.RequestFunction;
import com.wowza.wms.stream.*;
import com.wowza.wms.timedtext.cea608.*;
import com.wowza.wms.vhost.IVHost;

public class ModuleClosedCaptionLive extends ModuleBase
{
	class CCCommandSet
	{
		public short[] ccCommands = null;
		public int cpos = 0;
		
		public boolean isEmpty()
		{
			return cpos >= ccCommands.length;
		}
	}
	
	class CCTextBlock
	{
		public String text = null;
		
		public CCTextBlock(String text)
		{
			this.text = text;
		}
	}
	
	class ClosedCaptionStreamHandler implements IMediaStreamCallback, IMediaStreamH264SEINotify
	{
		private List<CCTextBlock> ccTextBlocks = new ArrayList<CCTextBlock>();
		private List<CCCommandSet> ccCommandSets = new ArrayList<CCCommandSet>();
		private int nalUnitLen = -1;
		private Object lock = new Object();
		private long lastDisplayTime = -1;
		private int maxDisplayTime = 10000;
		private int commandsPerFrame = ClosedCaptionCEA608Utils.MAX_CCCOUNT;
		private int payloadType = ClosedCaptionCEA608Utils.SEI_PAYLOADTYPE;
		private int channel = ClosedCaptionCEA608Utils.CHANNEL1;
		private int charsPerLine = ClosedCaptionCEA608Utils.COUNT_COLS+1;
		private int color = ClosedCaptionCEA608Utils.COL0_WHITE;
		private String characterSet = "UTF-8";
		
		// Called for each onTextData event in the stream
		public void onCallback(IMediaStream stream, RequestFunction function, AMFDataList params)
		{
			
			CCTextBlock ccTextBlock = null;
			while(true)
			{
				String text = getTextData(params);
				if (text == null)
					break;
				
				ccTextBlock = new CCTextBlock(text);
				break;
			}
			
			if (ccTextBlock != null)
			{
				if (logOnTextDataEvents)
					getLogger().info("ModuleClosedCaptionLive#OnTextDataHandler.onCallback["+stream.getContextStr()+"]: ccTextBlock:"+ccTextBlock.text);

				synchronized(lock)
				{
					ccTextBlocks.add(ccTextBlock);
				}
			}
		}
		
		// Called for each frame to enable insertion of SEI data. In this module we are inserting CAE-608/CAE-708 data into each frame as needed
		public void onVideoH264Packet(IMediaStream stream, AMFPacket packet, H264SEIMessages seiMessages)
		{
			
			// return if not an H.264 video stream
			int videoCodec = FLVUtils.getVideoCodec(packet);
			if (videoCodec != IVHost.CODEC_VIDEO_H264)
			{
				synchronized(lock)
				{
					ccTextBlocks.clear();
				}
				return;
			}

			if (removeExistingCEA608)
			{
				removeCEA608Captions(seiMessages);
			}

			// grab the most recent text events
			List<CCTextBlock> localOnTextDataEvents = null;
			synchronized(lock)
			{
				if (ccTextBlocks.size() > 0)
				{
					localOnTextDataEvents = new ArrayList<CCTextBlock>();
					localOnTextDataEvents.addAll(ccTextBlocks);
					ccTextBlocks.clear();
				}
			}
			
			// process new text events and format as CAE-608 commands
			if (localOnTextDataEvents != null)
			{
				Iterator<CCTextBlock> iter = localOnTextDataEvents.iterator();
				while(iter.hasNext())
				{
					CCTextBlock ccTextBlock = iter.next();
					
					List<String> lines = breakTextIntoLines(ccTextBlock.text);
					
					CCCommandSet ccCommandSet = getCAE608Commands(lines, characterSet, channel);
					if (ccCommandSet == null)
						break;
					
					synchronized(lock)
					{
						ccCommandSets.add(ccCommandSet);
					}
				}
			}
			
			// get the first CCCommandSet from the list
			CCCommandSet ccCommandSet = null;
			synchronized(lock)
			{
				if (ccCommandSets.size() > 0)
					ccCommandSet = ccCommandSets.get(0);
			}
			
			long currTime = System.currentTimeMillis();
			
			if (ccCommandSet != null)
			{
				int frameType = getFrameType(stream, packet);
				if (frameType != FLVUtils.FLV_DFRAME) // do not send CC data if a B-frame - out of order frames cause problems with CC processing
				{
					lastDisplayTime = -1;
					
					int commandCount = ccCommandSet.ccCommands.length-ccCommandSet.cpos;
					
					if (commandCount > commandsPerFrame)
						commandCount = commandsPerFrame;
					
					// format the next block of CC commands as SEI data and insert into the stream
					byte[] sbuf = getClosedCaption(ccCommandSet.ccCommands, ccCommandSet.cpos, commandCount);
					if (sbuf != null)
					{
						ccCommandSet.cpos += commandCount;
						seiMessages.setModified(true);
						seiMessages.addMessage(new H264SEIMessage(payloadType, sbuf, 0, sbuf.length));
					}				
				}

				// if each the end of the CCCommandSet then remove from the list
				if (ccCommandSet.isEmpty())
				{
					synchronized(lock)
					{
						ccCommandSets.remove(ccCommandSet);
						if (ccCommandSets.size() <= 0)
							lastDisplayTime = currTime;
					}
				}
			}
			
			// track how long a CC block has been on the screen and if longer than maxDisplayTime clear it from the screen
			if (maxDisplayTime > 0 && lastDisplayTime >= 0)
			{
				if ((currTime - lastDisplayTime) >= maxDisplayTime)
				{
					lastDisplayTime = -1;
					
					short ccCommands[] = new short[2];

					ccCommands[0] = ClosedCaptionCEA608Utils.CONTROLCODES_ENM[channel]; // ENM Erase Non-Displayed Memory
					ccCommands[1] = ClosedCaptionCEA608Utils.CONTROLCODES_EDM[channel]; // EDM Erase Displayed Memory
					
					// send clear screen commands
					byte[] sbuf = getClosedCaption(ccCommands, 0, ccCommands.length);
					if (sbuf != null)
					{
						seiMessages.setModified(true);
						seiMessages.addMessage(new H264SEIMessage(payloadType, sbuf, 0, sbuf.length));
					}				
				}
			}
		}
		

		// add text block to the CC list
		public void addClosedCaptionText(String text)
		{
			CCTextBlock ccTextBlock = new CCTextBlock(text);

			getLogger().info("ModuleClosedCaptionLive.addClosedCaptionText: ccTextBlock:"+ccTextBlock.text);

			synchronized(lock)
			{
				ccTextBlocks.add(ccTextBlock);
			}
		}


		// See http://en.wikipedia.org/wiki/CEA-708
		// format a set of closed caption events as SEI data
		private byte[] getClosedCaption(short[] cc_commands, int offset, int len)
		{
			int spos = 0;
			byte[] sbuf = new byte[ClosedCaptionCEA608Utils.SEI_STARTCODE.length+3+(len*3)+1];

			while(true)
			{
				System.arraycopy(ClosedCaptionCEA608Utils.SEI_STARTCODE, 0, sbuf, spos, ClosedCaptionCEA608Utils.SEI_STARTCODE.length);
				spos += ClosedCaptionCEA608Utils.SEI_STARTCODE.length;
				
				sbuf[spos] = ClosedCaptionCEA608Utils.SEI_USERDATATYPECODE;
				spos++;
				
				// Bits from high to low:
				// 7   : process em data
				// 6   : process cc data
				// 5   : additional_data_flag: if set then ATSC_reserved_user_data at end of packet
				// 0-4 : cc_count
				sbuf[spos] = (byte)(0x80 | 0x40 | len);
				spos++;
						
				// em_data (not currently parsed)
				sbuf[spos] = (byte)0x0FF; // reserved 8 1111 1111
				spos++;
				
				for(int cc=0;cc<len;cc++)
				{
					// Bits from high to low:
					// 3-7 : marker bits (all 1's)
					// 2   : cc_valid
					// 0-1 : cc_type
					sbuf[spos] = (byte)0xfc;
					spos++;
					
					sbuf[spos] = ClosedCaptionCEA608Utils.calcParity((byte)((cc_commands[offset+cc] >> 8) & 0x0FF));
					spos++;
					sbuf[spos] = ClosedCaptionCEA608Utils.calcParity((byte)((cc_commands[offset+cc] >> 0) & 0x0FF));
					spos++;
				}
				
				// marker bits (all 1's)
				sbuf[spos] = (byte)0x0FF; // marker 8 '1111 1111'
				spos++;
				
				// ATSC would be here if additional_data_flag was set
				
				
				break;
			}
						
			return sbuf;
		}

		// get the frame type of an H.264 frame - need to skip B-frames
		private int getFrameType(IMediaStream stream, AMFPacket packet)
		{
			int frameType = FLVUtils.getVideoFrameType(packet);
			if (frameType == FLVUtils.FLV_PFRAME && packet.getSize() > 5) // need to really dig into H.264 to see if B-frame
			{
				byte[] buffer = packet.getData();
				int boffset = 5;
				
				if (nalUnitLen < 0)
					nalUnitLen = getNALUnitSizeLength(stream, packet.getAbsTimecode());
									
				frameType = H264Utils.extractFrameType(buffer, boffset, buffer.length-boffset, nalUnitLen);
			}
			
			return frameType;
		}
		
		// get the nalUnitLen from the SPS data
		private int getNALUnitSizeLength(IMediaStream stream, long timecode)
		{
			int ret = H264Utils.DEFAULT_NALUNITSIZE_LENGTH;
			
			while(true)
			{
				AMFPacket packet = stream.getVideoCodecConfigPacket(timecode);
				if (packet == null)
					break;
				
				byte[] buffer = packet.getData();
				int size = packet.getSize();
				if (buffer == null || size <= 5)
					break;
				
				H264CodecConfigInfo codecConfigInfo = H264Utils.decodeAVCC(buffer, 5);
				if (codecConfigInfo == null)
					break;

				ret = codecConfigInfo.nalUnitLen;
				break;
			}
			
			return ret;
		}

		// extract text data from an onTextData event
		private String getTextData(AMFDataList onTextDataEvent)
		{
			String ret = null;
			
			while(true)
			{
				if (onTextDataEvent.size() < 2)
					break;
				
				int objIndex = 1;
				
				if (onTextDataEvent.getType(objIndex) != AMFData.DATA_TYPE_OBJECT && onTextDataEvent.getType(objIndex) != AMFData.DATA_TYPE_MIXED_ARRAY)
					break;
				
				AMFDataObj values = onTextDataEvent.getObject(objIndex);
				
				ret = values.getString("text");
				break;
			}

			return ret;
		}
		
		private List<String> breakTextIntoLines(String text)
		{
			List<String> allLines = new ArrayList<String>();
			
			List<String> brokenLines = breakOnHtmlBreaks(text);
			for (String broken : brokenLines)
			{
				List<String> lines = breakLongTextIntoLines(broken, charsPerLine);
				allLines.addAll(lines);
			}
			return allLines;
		}

		private List<String> breakOnHtmlBreaks(String originalText)
		{
			List<String> ret = new ArrayList<String>();
			
			if (StringUtils.isEmpty(originalText))
				return ret;
			
			try
			{
				String[] strings = originalText.split("\n");
				for (String string : strings)
				{
					ret.add(string);
				}	
			} catch (Exception e)
			{
				ret = new ArrayList<String>();
				ret.add(originalText);
			}

			return ret;
		}

		// break a string into lines
		// Note this method also exists in TimedTextUtils but local copy kept here because
		// that utility class did not exist when ModuleOnTextDataToCEA608 was created
		private List<String> breakLongTextIntoLines(String text, int charsPerLine)
		{
			List<String> lines = new ArrayList<String>();
			
			text = text.trim();
			while(true)
			{
				String line = text;
				if (text.length() > charsPerLine)
				{
					boolean didBreak = false;
					for(int i=charsPerLine;i>=0;i--)
					{
						if (Character.isWhitespace(text.charAt(i)))
						{
							didBreak = true;
							line = text.substring(0, i);
							text = text.substring(i+1);
							break;
						}
					}
					
					// hard break if no spaces
					if (!didBreak)
					{
						line = text.substring(0, charsPerLine);
						text = text.substring(charsPerLine);
					}
				}
				else
					text = "";
				
				lines.add(line.trim());
								
				text = text.trim();
				if (text.length() <= 0)
					break;
			}
			
			return lines;
		}
		

		private byte[] charToByte(String text, int pos, String characterSet, int channel)
		{
			return ClosedCaptionCEA608Utils.charUTF8ToCAE608(text, pos, characterSet, channel);
		}
		// count the number of characters total and include end of line padding
		private int getCharacterCount(List<String> lines, String characterSet, int channel)
		{
			int charCount = 0;
			
			Iterator<String> iter = lines.iterator();
			while(iter.hasNext())
			{
				String line = iter.next();
				
				int j=0;
				for(int c=0;c<line.length();c++)
				{
					byte[] charByte = charToByte(line, c, characterSet, channel);
					
					for(int ci=0;ci<charByte.length;ci++)
					{
						if ((charByte[ci] & 0x0ff) < 0x20 && (j%2) > 0)
							j++;
						
						j++;
					}
				}
				
				if ((j%2) > 0)
					j++;
				
				charCount += j;
			}
			
			return charCount;
		}
		
		// convert line into CC commands
		private CCCommandSet getCAE608Commands(List<String> lines, String characterSet, int channel)
		{
			int prefixSize = 3;
			int perLineSize = 2;
			int postfixSize = 2;
			
			CCCommandSet ccCommandSet = new CCCommandSet();
			
			int charCount = getCharacterCount(lines, characterSet, channel);
			
			int size = prefixSize + (perLineSize*lines.size()) + (charCount/2) + postfixSize; // + 1;
			
			ccCommandSet.ccCommands = new short[size];
			
			int row = 0; // initial row
			
			int cpos = 0;
			
			// prefix
			ccCommandSet.ccCommands[cpos] = ClosedCaptionCEA608Utils.CONTROLCODES_ENM[channel]; cpos++; // ENM Erase Non-Displayed Memory
			ccCommandSet.ccCommands[cpos] = ClosedCaptionCEA608Utils.CONTROLCODES_RCL[channel]; cpos++; // RCL Resume caption loading
			ccCommandSet.ccCommands[cpos] = (short)(ClosedCaptionCEA608Utils.COL0_HIBYTE[channel][row] | ClosedCaptionCEA608Utils.COL0_LOWBYTE[color][row]); cpos++; // Set the caption color
			
			// bottom justify
			int rowOffset = ClosedCaptionCEA608Utils.COUNT_ROWS-lines.size();
			if (rowOffset < 0)
				rowOffset = 0;
			
			for(int i=0;i<Math.min(lines.size(), ClosedCaptionCEA608Utils.COUNT_ROWS);i++)
			{				
				// perline
				ccCommandSet.ccCommands[cpos] = (short)(0x0FFFF & (ClosedCaptionCEA608Utils.COL0_HIBYTE[channel][rowOffset+i] | ClosedCaptionCEA608Utils.COL0_LOWBYTE[color][rowOffset+i])); cpos++; // set position and color
				ccCommandSet.ccCommands[cpos] = ClosedCaptionCEA608Utils.CONTROLCODES_BAS[0];  cpos++; // BAS Background Black, Semi-transparent
									
				String line = lines.get(i);
				//byte[] lineBytes = line.getBytes();
								
				int j=0;
				for(int c=0;c<line.length();c++)
				{
					byte[] charByte = charToByte(line, c, characterSet, channel);
					
					for(int ci=0;ci<charByte.length;ci++)
					{
						// be sure two byte codes start on word boundaries
						if ((charByte[ci] & 0x0ff) < 0x20 && (j%2) > 0)
						{
							ccCommandSet.ccCommands[cpos] |= 0x80; j++;
							cpos++;
						}
						
						if ((j%2) <= 0)
						{
							ccCommandSet.ccCommands[cpos] = charByte[ci]; j++;
							ccCommandSet.ccCommands[cpos] <<= 8;
						}
						else
						{
							ccCommandSet.ccCommands[cpos] |= charByte[ci]; j++;
							cpos++;
						}
					}
				}
				
				// padding
				if ((j%2) > 0)
				{
					ccCommandSet.ccCommands[cpos] |= 0x80;
					cpos++;
				}
			}
			
			// postfix
			ccCommandSet.ccCommands[cpos] = ClosedCaptionCEA608Utils.CONTROLCODES_EDM[channel]; cpos++; // EDM Erase Displayed Memory
			ccCommandSet.ccCommands[cpos] = ClosedCaptionCEA608Utils.CONTROLCODES_EOC[channel]; cpos++; // EOC End of Caption (Flip Memories)
			
			return ccCommandSet;
		}

		public int getMaxDisplayTime()
		{
			return maxDisplayTime;
		}

		public void setMaxDisplayTime(int maxDisplayTime)
		{
			this.maxDisplayTime = maxDisplayTime;
		}

		public int getCommandsPerFrame()
		{
			return commandsPerFrame;
		}

		public void setCommandsPerFrame(int commandsPerFrame)
		{
			this.commandsPerFrame = commandsPerFrame;
		}

		public int getChannel()
		{
			return channel;
		}

		public void setChannel(int channel)
		{
			this.channel = channel;
		}

		public int getColor()
		{
			return color;
		}

		public void setColor(int color)
		{
			this.color = color;
		}

		public String getCharacterSet()
		{
			return characterSet;
		}

		public void setCharacterSet(String characterSet)
		{
			this.characterSet = characterSet;
		}

	}
	
	private IApplicationInstance appInstance = null;
	private int maxDisplayTime = 10000;
	private int commandsPerFrame = ClosedCaptionCEA608Utils.MAX_CCCOUNT;
	private int channel = ClosedCaptionCEA608Utils.CHANNEL1;
	private int color = ClosedCaptionCEA608Utils.COL0_WHITE;
	private String characterSet = "UTF-8";
	private boolean logOnTextDataEvents = false;
	private boolean removeExistingCEA608 = false;
	
	// onAppStart - get settings
	public void onAppStart(IApplicationInstance appInstance)
	{
		this.appInstance = appInstance;
		
		WMSProperties props = appInstance.getProperties();
		
		maxDisplayTime = props.getPropertyInt("closedCaptionLiveMaxDisplayTime", maxDisplayTime);
		commandsPerFrame = props.getPropertyInt("closedCaptionLiveCommandsPerFrame", commandsPerFrame);
		channel = props.getPropertyInt("closedCaptionLiveChannel", channel);
		color = props.getPropertyInt("closedCaptionLiveColor", color);
		characterSet = props.getPropertyStr("closedCaptionLiveCharacterSet", characterSet);
		logOnTextDataEvents = props.getPropertyBoolean("closedCaptionLiveLogOnTextDataEvents", logOnTextDataEvents);
		removeExistingCEA608 = props.getPropertyBoolean("closedCaptionLiveRemoveExistingCEA608", removeExistingCEA608);
		
		getLogger().info("ModuleClosedCaptionLive.onAppStart["+appInstance.getContextStr()+"] logOnTextDataEvents:"+logOnTextDataEvents);
	}

	// onAppStop
	public void onAppStop(IApplicationInstance appInstance)
	{
		getLogger().info("ModuleClosedCaptionLive.onAppStop["+appInstance.getContextStr()+"]");
	}
	
	// onStreamCreate - hook up handlers
	public void onStreamCreate(IMediaStream stream)
	{
		ClosedCaptionStreamHandler closedCaptionStreamHandler = new ClosedCaptionStreamHandler();
		
		closedCaptionStreamHandler.setMaxDisplayTime(maxDisplayTime);
		closedCaptionStreamHandler.setCommandsPerFrame(commandsPerFrame);
		closedCaptionStreamHandler.setChannel(channel);
		closedCaptionStreamHandler.setColor(color);
		closedCaptionStreamHandler.setCharacterSet(characterSet);
		
		stream.getProperties().setProperty("ClosedCaptionStreamHandler", closedCaptionStreamHandler);

		stream.registerCallback("onTextData", closedCaptionStreamHandler);
		stream.addVideoH264SEIListener(closedCaptionStreamHandler);
	}
	
	// add closed caption text to the next set of frames
	public void addClosedCaptionText(IMediaStream stream, String text)
	{
		while(true)
		{
			Object obj = stream.getProperties().getProperty("ClosedCaptionStreamHandler");
			if (obj == null)
			{
				getLogger().warn("ModuleClosedCaptionLive.addClosedCaptionText["+stream.getContextStr()+"] ClosedCaptionStreamHandler is missing.");
				break;
			}
			
			if (!(obj instanceof ClosedCaptionStreamHandler))
				break;
			
			ClosedCaptionStreamHandler closedCaptionStreamHandler = (ClosedCaptionStreamHandler)obj;

			closedCaptionStreamHandler.addClosedCaptionText(text);
			break;
		}
	}
	
	// These are temporarily copied here since CEA608SEIMessageUtils is not available in 3.5
	private boolean isCEA608Caption(H264SEIMessage message)
	{
		if (message.getPayloadType() != 4)
			return false;
		byte[] buffer = message.getPayloadBuffer();
		int offset = message.getPayloadOffset();
	
		if (buffer[offset + 2] == '1' && buffer[offset + 3] == 'G' && buffer[offset + 4] == 'A' && buffer[offset + 5] == '9' && buffer[offset + 6] == '4')
			return true;
	
		return false;
	}

	private void removeCEA608Captions(H264SEIMessages seiMessages)
	{
		if (seiMessages == null || seiMessages.isEmpty())
			return;
		
		int count = seiMessages.getMessageCount();
		// Start at top and work down so array shifting doesn't mess up indexes
		for (int i = count-1; i >= 0; i--)
		{
			H264SEIMessage msg = seiMessages.getMessage(i);
			if (isCEA608Caption(msg))
			{
				seiMessages.removeMessage(i);
				seiMessages.setModified(true);
		}
		}
	}

}
