/**
 * Wowza Media Server and all components Copyright 2006 - 2013, Wowza Media Systems, LLC, licensed pursuant to the Wowza Media Software End User License Agreement.
 */
package com.wowza.wms.plugin.collection.module;

import com.wowza.wms.amf.*;
import com.wowza.wms.application.*;
import com.wowza.wms.client.*;
import com.wowza.wms.httpstreamer.model.IHTTPStreamerSession;
import com.wowza.wms.request.*;
import com.wowza.wms.rtp.model.RTPSession;
import com.wowza.wms.rtp.model.RTPUrl;
import com.wowza.wms.stream.*;
import com.wowza.wms.stream.mediacaster.MediaStreamMediaCasterUtils;
import com.wowza.wms.util.ModuleUtils;
import com.wowza.wms.module.*;

public class ModuleLimitStreamViewers extends ModuleBase
{
	static final public int MAXVIEWERS = 200;

	private int maxStreamViewers = MAXVIEWERS;
	private IApplicationInstance appInstance;
	private boolean logConnectionCounts = true;
	private boolean logRejections = true;
	
	public void onAppStart(IApplicationInstance appInstance)
	{
		this.appInstance = appInstance;
		this.maxStreamViewers = appInstance.getProperties().getPropertyInt("maxStreamViewers", MAXVIEWERS);
		this.logConnectionCounts = appInstance.getProperties().getPropertyBoolean("limitStreamViewersLogConnectionCounts", this.logConnectionCounts);
		this.logRejections = appInstance.getProperties().getPropertyBoolean("limitStreamViewersLogRejections", this.logRejections);
		
		getLogger().info("ModuleLimitStreamViewers limit: " + this.maxStreamViewers + " logConnections: " + this.logConnectionCounts);
	}

	public void play(IClient client, RequestFunction function, AMFDataList params)
	{
		String streamName = params.getString(PARAM1);

		//get the real stream name if this is an alias.
		streamName = ((ApplicationInstance)appInstance).internalResolvePlayAlias(streamName, client);

		int count = getViewerCounts(streamName, client);

		if (count < this.maxStreamViewers)
		{
			this.invokePrevious(client, function, params);
		}
		else
		{
			IMediaStream stream = getStream(client, function);
			if (stream != null)
			{
				String code = "NetStream.Play.Failed";
				String description = "ModuleLimitViewers: Over viewer limit[" + this.maxStreamViewers + "]";
				
				sendStreamOnStatusError(stream, code, description);

				if(logRejections)
					getLogger().info("ModuleLimitViewers: Over viewer limit[" + this.maxStreamViewers + " streamName: "+ streamName + "]");
			}
		}
	}

	public void onHTTPSessionCreate(IHTTPStreamerSession httpSession)
	{
		String streamName = httpSession.getStreamName();
		// No need to get the alias here as it has already been done so just need to get the connection count.
		int count = getViewerCounts(streamName);
		// When an HTTP session is created, the stream is already allocated here so will register as one http count.
		if(count > this.maxStreamViewers)
		{
			httpSession.rejectSession();
			if(logRejections)
				getLogger().info("ModuleLimitViewers: Over viewer limit[" + this.maxStreamViewers + " streamName: "+ streamName + "]");
		}
	}
	
	public void onRTPSessionCreate(RTPSession rtpSession)
	{
		String uri = rtpSession.getUri();
		RTPUrl url = new RTPUrl(uri);
		String streamName = url.getStreamName();
		
		streamName = ((ApplicationInstance)appInstance).internalResolvePlayAlias(streamName, rtpSession);
		
		int count = getViewerCounts(streamName);
		
		if(count >= this.maxStreamViewers)
		{
			rtpSession.rejectSession();
			if(logRejections)
				getLogger().info("ModuleLimitViewers: Over viewer limit[" + this.maxStreamViewers + " streamName: "+ streamName + "]");
		}
	}
	
	private int getViewerCounts(String streamName)
	{
		return getViewerCounts(streamName, null);
	}
	
	private synchronized int getViewerCounts(String streamName, IClient client)
	{
		int count  = 0;
		String streamExt = MediaStream.BASE_STREAM_EXT;
		int rtmpCount = 0;
		int httpCount = 0;
		int rtpCount = 0;
		
		if(streamName != null)
		{
			String[] streamDecode = ModuleUtils.decodeStreamExtension(streamName, streamExt);
			streamName = streamDecode[0];
			streamExt = streamDecode[1];

			boolean isStreamNameURL = streamName.indexOf("://") >= 0;
			int streamQueryIdx = streamName.indexOf("?");
			if (!isStreamNameURL && streamQueryIdx >= 0)
			{
				streamName = streamName.substring(0, streamQueryIdx);
			}
			
			rtmpCount += appInstance.getPlayStreamCount(streamName);
			httpCount += appInstance.getHTTPStreamerSessionCount(streamName);
			rtpCount += appInstance.getRTPSessionCount(streamName);
			
			// Test for mediaCaster streams like wowz://[origin-ip]:1935/origin/myStream.
			String mediaCasterName = MediaStreamMediaCasterUtils.mapMediaCasterName(appInstance, client, streamName);
			if(!mediaCasterName.equals(streamName))
			{
				getLogger().info("ModuleLimitStreamViewers.getViewerCounts matching mediaCaster name: " + mediaCasterName);
				rtmpCount += appInstance.getPlayStreamCount(mediaCasterName);
				httpCount += appInstance.getHTTPStreamerSessionCount(mediaCasterName);
				rtpCount += appInstance.getRTPSessionCount(mediaCasterName);
			}
			count = rtmpCount + httpCount + rtpCount;
			
			if(logConnectionCounts)
				getLogger().info("ModuleLimitStreamViewers.getViewerCounts streamName: " + streamName + " total:" + count + " rtmp: " + rtmpCount + " http: " + httpCount + " rtp: " + rtpCount);

		}
		return count;
	}
}
