/**
 * Wowza Media Server and all components Copyright 2006 - 2013, Wowza Media Systems, LLC, licensed pursuant to the Wowza Media Software End User License Agreement.
 */
package com.wowza.wms.plugin.collection.module;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.wowza.wms.application.IApplicationInstance;
import com.wowza.wms.mediacaster.IMediaCaster;
import com.wowza.wms.mediacaster.IMediaCasterNotify2;
import com.wowza.wms.module.ModuleBase;
import com.wowza.wms.stream.IMediaStream;
import com.wowza.wms.stream.IMediaStreamActionNotify;
import com.wowza.wms.stream.IMediaStreamPlay;
import com.wowza.wms.stream.publish.PlaylistItem;
import com.wowza.wms.stream.publish.Stream;

public class ModuleLoopUntilLive extends ModuleBase
{
	
	private class StreamListener implements IMediaStreamActionNotify
	{
		public void onPublish(IMediaStream stream, String streamName, boolean isRecord, boolean isAppend)
		{
			String[] names = liveStreamNames.split(",");
			for(String name : names)
			{
				if (name.trim().equalsIgnoreCase(streamName))
				{
					if (appInstance.getMediaCasterStreams().getMediaCaster(streamName) != null && handleMediaCasters)
					{
						getLogger().info("ModuleLoopUntilLive.StreamListener.onPublish Stream is MediaCaster. Swapping handled by MediaCasterListener.onStreamStart [" + streamName +"]");
						return;
					}
					getLogger().info("ModuleLoopUntilLive.StreamListener.onPublish Swapping to live [" + streamName +"]");
					swapToLive(name);
					break;
				}
			}
		}

		public void onUnPublish(IMediaStream stream, String streamName, boolean isRecord, boolean isAppend)
		{
			String[] names = liveStreamNames.split(",");
			for(String name : names)
			{
				if (name.trim().equalsIgnoreCase(streamName))
				{
					if (appInstance.getMediaCasterStreams().getMediaCaster(streamName) != null && handleMediaCasters)
					{
						getLogger().info("ModuleLoopUntilLive.onUnPublish Stream is MediaCaster. Swapping handled by MediaCasterListener.onStreamStop [" + streamName +"]");
						return;
					}
					getLogger().info("ModuleLoopUntilLive.StreamListener.onPublish Swapping to playlist [" + streamName +"]");
					swapToPlaylist(name);
					break;
				}
			}
		}

		public void onPause(IMediaStream stream, boolean isPause, double location)
		{
		}

		public void onPlay(IMediaStream stream, String streamName, double playStart, double playLen, int playReset)
		{
		}

		public void onSeek(IMediaStream stream, double location)
		{
		}

		public void onStop(IMediaStream stream)
		{
		}
	}
	
	private class MediaCasterListener implements IMediaCasterNotify2
	{

		public void onMediaCasterCreate(IMediaCaster mediaCaster)
		{
		}

		public void onMediaCasterDestroy(IMediaCaster mediaCaster)
		{
		}

		public void onRegisterPlayer(IMediaCaster mediaCaster, IMediaStreamPlay player)
		{
		}

		public void onUnRegisterPlayer(IMediaCaster mediaCaster, IMediaStreamPlay player)
		{
		}

		public void onSetSourceStream(IMediaCaster mediaCaster, IMediaStream stream)
		{
		}

		public void onConnectStart(IMediaCaster mediaCaster)
		{
		}

		public void onConnectSuccess(IMediaCaster mediaCaster)
		{
		}

		public void onConnectFailure(IMediaCaster mediaCaster)
		{
		}

		public void onStreamStart(IMediaCaster mediaCaster)
		{
			String streamName = mediaCaster.getStream().getName();
			String[] names = liveStreamNames.split(",");
			for(String name : names)
			{
				if (name.trim().equalsIgnoreCase(streamName))
				{
					getLogger().info("ModuleLoopUntilLive.MediaCasterListener.onStreamStop Swapping to live [" + streamName +"]");
					swapToLive(name);
					break;
				}
			}
		}

		public void onStreamStop(IMediaCaster mediaCaster)
		{
			String streamName = mediaCaster.getStream().getName();
			String[] names = liveStreamNames.split(",");
			for(String name : names)
			{
				if (name.trim().equalsIgnoreCase(streamName))
				{
					getLogger().info("ModuleLoopUntilLive.MediaCasterListener.onStreamStop Swapping to playlist [" + streamName +"]");
					swapToPlaylist(name);
					break;
				}
			}
		}
		
	}

	private IApplicationInstance appInstance;
	private String liveStreamNames ="myStream";
	private String outStreamNames = "Stream1";
	private boolean reloadEntirePlaylist = true;
	private boolean handleMediaCasters = true;
	private IMediaStreamActionNotify actionNotify = new StreamListener();

	private Map<String, List<PlaylistItem>> playlists = new HashMap<String, List<PlaylistItem>>();
	private Map<String, Integer> playlistIndexes = new HashMap<String, Integer>();
	
	public void onAppStart(IApplicationInstance appInstance) {
		init(appInstance);
		getLogger().info("ModuleLoopUntilLive onAppStart");
	}
	
	public void init(IApplicationInstance appInstance)
	{
		this.appInstance = appInstance;
		appInstance.addMediaCasterListener(new MediaCasterListener());
		
		this.liveStreamNames = appInstance.getProperties().getPropertyStr("loopUntilLiveStream", liveStreamNames);
		this.outStreamNames = appInstance.getProperties().getPropertyStr("loopUntilLiveOutStream", outStreamNames);
		this.reloadEntirePlaylist = appInstance.getProperties().getPropertyBoolean("loopUntilLiveReloadEntirePlaylist", reloadEntirePlaylist);
		this.handleMediaCasters = appInstance.getProperties().getPropertyBoolean("loopUntilLiveHandleMediaCasters", handleMediaCasters);
	}

	public void onStreamCreate(IMediaStream stream)
	{
		if(this.appInstance == null)
			init(stream.getStreams().getAppInstance());
		stream.addClientListener(actionNotify);
	}

	public void onStreamDestroy(IMediaStream stream)
	{
		stream.removeClientListener(actionNotify);
	}

	private void swapToLive(String streamName)
	{
		String[] liveNames = liveStreamNames.split(",");
		String[] outNames = outStreamNames.split(",");
		int idx = 0;
		
		while (idx < liveNames.length)
		{
			String liveName = liveNames[idx].trim();
			if(streamName.equalsIgnoreCase(liveName))
			{
				if(outNames.length > idx)
				{
					String outName = outNames[idx].trim();
					
					Stream stream = (Stream)appInstance.getProperties().get(outName);
					if(stream != null)
					{
						List<PlaylistItem> playlist = stream.getPlaylist();
						int currentItemIndex = stream.getCurrentItem().getIndex();
						playlists.put(outName, playlist);
						playlistIndexes.put(outName, currentItemIndex);
						stream.play(liveName, -2, -1, true);
						getLogger().info("ModuleLoopUntilLive.swapToLive [" + stream.getName() + "]");
					}
				}
			}
			idx++;
		}
	}
	
	private void swapToPlaylist(String streamName)
	{
		String[] liveNames = liveStreamNames.split(",");
		String[] outNames = outStreamNames.split(",");
		int idx = 0;
		
		while (idx < liveNames.length)
		{
			String liveName = liveNames[idx].trim();
			if(streamName.equalsIgnoreCase(liveName))
			{
				if(outNames.length > idx)
				{
					String outName = outNames[idx].trim();
					Stream stream = (Stream)appInstance.getProperties().get(outName);
					if(stream != null)
					{
						List<PlaylistItem> playlist = playlists.remove(outName);
						int currentItemIndex = playlistIndexes.remove(outName);
						
						if(reloadEntirePlaylist)
						{
							boolean reset = true;
							for(PlaylistItem item : playlist)
							{
								stream.play(item.getName(), item.getStart(), item.getLength(), reset);
								reset = false;
							}
							stream.play(currentItemIndex + 1);
						}
						else
						{
							if(playlist.size() > currentItemIndex)
							{
								PlaylistItem item = playlist.get(currentItemIndex);
								stream.play(item.getName(), item.getStart(), item.getLength(), true);
							}
						}
						getLogger().info("ModuleLoopUntilLive.swapToPlaylist [" + stream.getName() + "]");
					}
				}
			}
			idx++;
		}
	}

	public String getLiveStreamNames()
	{
		return liveStreamNames;
	}

	public void setLiveStreamNames(String liveStreamNames)
	{
		this.liveStreamNames = liveStreamNames;
	}

	public String getOutStreamNames()
	{
		return outStreamNames;
	}

	public void setOutStreamNames(String outStreamNames)
	{
		this.outStreamNames = outStreamNames;
	}

	public boolean isReloadEntirePlaylist()
	{
		return reloadEntirePlaylist;
	}

	public void setReloadEntirePlaylist(boolean reloadEntirePlaylist)
	{
		this.reloadEntirePlaylist = reloadEntirePlaylist;
	}

	public boolean isHandleMediaCasters()
	{
		return handleMediaCasters;
	}

	public void setHandleMediaCasters(boolean handleMediaCasters)
	{
		this.handleMediaCasters = handleMediaCasters;
	}
}
