/**
 * Wowza Media Server and all components Copyright 2006 - 2013, Wowza Media Systems, LLC, licensed pursuant to the Wowza Media Software End User License Agreement.
 */
package com.wowza.wms.plugin.collection.serverlistener;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.wowza.util.StringUtils;
import com.wowza.wms.application.IApplication;
import com.wowza.wms.application.IApplicationInstance;
import com.wowza.wms.application.IApplicationInstanceNotify;
import com.wowza.wms.application.WMSProperties;
import com.wowza.wms.logging.WMSLogger;
import com.wowza.wms.logging.WMSLoggerFactory;
import com.wowza.wms.server.IServer;
import com.wowza.wms.server.IServerNotify2;
import com.wowza.wms.server.Server;
import com.wowza.wms.stream.publish.IStreamActionNotify;
import com.wowza.wms.stream.publish.Playlist;
import com.wowza.wms.stream.publish.PlaylistItem;
import com.wowza.wms.stream.publish.Publisher;
import com.wowza.wms.stream.publish.Stream;
import com.wowza.wms.vhost.IVHost;
import com.wowza.wms.vhost.VHostSingleton;

public class ServerListenerStreamPublisher implements IServerNotify2
{

	private class ScheduledItem
	{
		private IApplicationInstance appInstance;
		private Timer timer;
		private Date start;
		private Playlist playlist;
		private Stream stream;

		public ScheduledItem(IApplicationInstance appInstance, Date start, Playlist playlist, Stream stream)
		{
			this.appInstance = appInstance;
			this.start = start;
			this.playlist = playlist;
			this.stream = stream;
			this.timer = new Timer();
		}

		/**
		 * Start a timer to load the schedule when it triggers.  After it triggers, remove the schedule from the schedulesMap.
		 */
		public void start()
		{
			timer.schedule(new TimerTask()
			{
				public void run()
				{
					playlist.open(stream);
					log.info("ServerListenerStreamPublisher Scheduled stream is now live: " + stream.getName());
					removeFromList();
					timer = null;
				}
			}, start);
			log.info("scheduled playlist: " + playlist.getName() + " on stream: " + stream.getName() + " for:" + start.toString());
		}

		/**
		 * Cancel the schedule and remove from schedulesMap.
		 */
		public void stop()
		{
			if (timer != null)
			{
				timer.cancel();
				log.info("cancelled playlist: " + playlist.getName() + " on stream: " + stream.getName() + " for:" + start.toString());
				removeFromList();
				timer = null;
			}
		}

		/**
		 * Remove schedule from schedulesMap as it no longer needs to be referenced.
		 */
		private void removeFromList()
		{
			synchronized(lock)
			{
				Map<String, List<ScheduledItem>> schedulesMap = (Map<String, List<ScheduledItem>>)appInstance.getProperties().get("streamPublisherSchedules");
				if(schedulesMap != null)
				{
					List<ScheduledItem> schedules = schedulesMap.get(stream.getName());
					if (schedules != null)
					{
						schedules.remove(this);
					}
				}
			}
		}
	}

	private class StreamListener implements IStreamActionNotify
	{
		private IApplicationInstance appInstance;

		StreamListener(IApplicationInstance appInstance)
		{
			this.appInstance = appInstance;
		}

		public void onPlaylistItemStart(Stream stream, PlaylistItem item)
		{
			try
			{
				String name = item.getName();
				appInstance.broadcastMsg("PlaylistItemStart", name);
				if(stream.isSwitchLog())
					WMSLoggerFactory.getLogger(null).info("ServerListenerStreamPublisher PlayList Item Start: " + name);
			}
			catch (Exception ex)
			{
				WMSLoggerFactory.getLogger(null).info("ServerListenerStreamPublisher Get Item error: " + ex.getMessage());
			}
		}

		public void onPlaylistItemStop(Stream stream, PlaylistItem item)
		{
			if (stream.getPlaylist().contains(item) && item.getIndex() == (stream.getPlaylist().size() - 1))
			{
				if (!stream.getRepeat())  //  Stream will be unpublished and shut down.  Any future schedules will be canceled.
				{
					Map<String, Stream> streams = (Map<String, Stream>)appInstance.getProperties().get("streamPublisherStreams");
					if(streams != null)
					{
						streams.remove(stream.getName());
					}
					shutdownStream(appInstance, stream);
					WMSLoggerFactory.getLogger(null).info("ServerListenerStreamPublisher: closing stream: " + stream.getName());
				}
			}
		}
	}

	private class AppInstanceListener implements IApplicationInstanceNotify
	{

		public void onApplicationInstanceCreate(IApplicationInstance appInstance)
		{
		}

		public void onApplicationInstanceDestroy(IApplicationInstance appInstance)
		{
			WMSProperties props = appInstance.getProperties();
			Map<String, Stream> streams = (Map<String, Stream>)props.remove("streamPublisherStreams");
			if(streams != null)
			{
				for(Stream stream : streams.values())
				{
					shutdownStream(appInstance, stream);
				}
			}
		}
	}

	public final static String PROP_STREAMPUBLISHER = "serverListenerStreamPublisher";

	private WMSLogger log = WMSLoggerFactory.getLogger(null);
	private Object lock = new Object();

	public void onServerCreate(IServer server)
	{
		server.getProperties().setProperty(PROP_STREAMPUBLISHER, this);
	}

	public void onServerInit(IServer server)
	{
		log.info("ServerListenerStreamPublisher Started.");
		IVHost vhost = null;
		IApplication application = null;
		IApplicationInstance appInstance = null;
		String vhostName = server.getProperties().getPropertyStr("PublishToVHost", "_defaultVHost_"); // Old Prop Name
		vhostName = server.getProperties().getPropertyStr("streamPublisherVHost", vhostName); // New Prop Name
		if (StringUtils.isEmpty(vhostName))
		{
			log.info("ServerListenerStreamPublisher: publishToVHost is empty. Can not run.");
			return;
		}
		try
		{
			vhost = VHostSingleton.getInstance(vhostName);
		}
		catch (Exception e)
		{
			log.info("ServerListenerStreamPublisher: Failed to get Vhost can not run.");
			return;
		}
		String appContext = server.getProperties().getPropertyStr("PublishToApplication", "live/_definst_"); // Old Prop Name
		appContext = server.getProperties().getPropertyStr("streamPublisherApplication", appContext); // New Prop Name
		if (StringUtils.isEmpty(appContext))
		{
			log.info("ServerListenerStreamPublisher: publishToApplication empty. Can not run.");
			return;
		}
		String[] appNameParts = appContext.split("/");
		String appName = appNameParts[0];
		String appInstName = appNameParts.length > 1 ? appNameParts[1] : IApplicationInstance.DEFAULT_APPINSTANCE_NAME;
		try
		{
			application = vhost.getApplication(appName);
		}
		catch (Exception e)
		{
			log.info("ServerListenerStreamPublisher: Failed to get Application can not run.");
			return;
		}
		AppInstanceListener listener = (AppInstanceListener)application.getProperties().get("streamPublisherAppInstanceListener");
		if (listener == null)
		{
			listener = new AppInstanceListener();
			application.addApplicationInstanceListener(listener);
			application.getProperties().setProperty("streamPublisherAppInstanceListener", listener);
		}
		try
		{
			appInstance = application.getAppInstance(appInstName);
		}
		catch (Exception e)
		{
			log.info("ServerListenerStreamPublisher: Failed to get Application Instance can not run.");
			return;
		}
		
		//  Module onAppStart runs as soon as getAppInstance() is called so check to see if the module loaded the schedule.
		if(appInstance.getProperties().getPropertyBoolean("streamPublisherScheduleLoaded", false))
		{
			log.info("ServerListenerStreamPublisher: Schedule loaded by module.");
		}
		else
		{
			String ret = loadSchedule(appInstance);
			log.info("ServerListenerStreamPublisher: "+ret);
		}
	}

	public String loadSchedule(IApplicationInstance appInstance)
	{
		WMSProperties serverProps = Server.getInstance().getProperties();
		WMSProperties props = appInstance.getProperties();
		String ret = "";
		synchronized(lock)
		{
			boolean switchLog = serverProps.getPropertyBoolean("streamPublisherSwitchLog", true);
			switchLog = props.getPropertyBoolean("streamPublisherSwitchLog", switchLog);
			boolean passMetaData = serverProps.getPropertyBoolean("PassthruMetaData", true); // Old Prop Name
			passMetaData = serverProps.getPropertyBoolean("streamPublisherPassMetaData", passMetaData); // New Prop Name
			// Allow override in Application.xml
			passMetaData = props.getPropertyBoolean("PassthruMetaData", passMetaData); // Old Prop Name
			passMetaData = props.getPropertyBoolean("streamPublisherPassMetaData", passMetaData); // New Prop Name
			// see if a schedule file is specified in the target application
			String scheduleSmil = serverProps.getPropertyStr("streamPublisherSmilFile", "streamschedule.smil");
			scheduleSmil = props.getPropertyStr("streamPublisherSmilFile", scheduleSmil);
			String storageDir = appInstance.getStreamStorageDir();
			try
			{
				String smilLoc = storageDir + "/" + scheduleSmil.replace("..", "");

				File playlistxml = new File(smilLoc);

				if (playlistxml.exists() == false)
				{
					return "Could not find playlist file: " + smilLoc;
				}
				DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

				DocumentBuilder db = null;
				Document document = null;
				try
				{

					db = dbf.newDocumentBuilder();
					document = db.parse("file:///" + smilLoc);

				}
				catch (Exception e)
				{
					return "XML Parse failed";
				}

				document.getDocumentElement().normalize();

				NodeList streams = document.getElementsByTagName("stream");
				Map<String, Stream> streamsList = (Map<String, Stream>)props.getProperty("streamPublisherStreams");
				if(streamsList == null)
				{
					streamsList = new HashMap<String, Stream>();
					props.setProperty("streamPublisherStreams", streamsList);
				}
				Map<String, Stream> tmp = new HashMap<String, Stream>();
				tmp.putAll(streamsList);
				streamsList.clear();
				Map<String, List<ScheduledItem>> schedulesMap = (Map<String, List<ScheduledItem>>)props.getProperty("streamPublisherSchedules");
				if(schedulesMap == null)
				{
					schedulesMap = new HashMap<String, List<ScheduledItem>>();
					props.setProperty("streamPublisherSchedules", schedulesMap);
				}
				for (int i = 0; i < streams.getLength(); i++)
				{
					Node streamItem = streams.item(i);
					if (streamItem.getNodeType() == Node.ELEMENT_NODE)
					{
						Element e = (Element)streamItem;
						String streamName = e.getAttribute("name");

						log.info("ServerListenerStreamPublisher: Stream name is '" + streamName + "'");

						Stream stream = tmp.get(streamName);
						if (stream == null)
						{
							stream = Stream.createInstance(appInstance, streamName);
							if (stream == null)
							{
								log.error("ServerListenerStreamPublisher cannot create stream: " + streamName);
								continue;
							}
							IStreamActionNotify actionNotify = (IStreamActionNotify)props.get("streamPublisherStreamListener");
							if(actionNotify == null)
							{
								actionNotify = new StreamListener(appInstance);
								props.setProperty("streamPublisherStreamListener", actionNotify);
							}
							stream.addListener(actionNotify);
						}
						tmp.remove(streamName);
						streamsList.put(streamName, stream);
						props.setProperty(streamName, stream);  // Required for ModuleLoopUntilLive.
					}
				}

				//  Shut down any streams still in tmp as they are not in the new schedule.
				for(Stream stream : tmp.values())
				{
					shutdownStream(appInstance, stream);
				}
				tmp.clear();
				
				//  Iterate through the existing streams for the application and remove any existing schedules.
				for(Stream stream : streamsList.values())
				{
					List<ScheduledItem> schedules = schedulesMap.remove(stream.getName());
					if(schedules != null)
					{
						for (ScheduledItem item : schedules)
						{
							item.stop();
						}
						schedules.clear();
					}
				}

				NodeList playList = document.getElementsByTagName("playlist");
				if (playList.getLength() == 0)
				{
					return "No playlists defined in smil file";
				}
				for (int i = 0; i < playList.getLength(); i++)
				{
					Node scheduledPlayList = playList.item(i);

					if (scheduledPlayList.getNodeType() == Node.ELEMENT_NODE)
					{
						Element e = (Element)scheduledPlayList;

						NodeList videos = e.getElementsByTagName("video");
						if (videos.getLength() == 0)
						{
							return "No videos defined in stream";
						}

						String streamName = e.getAttribute("playOnStream");
						if (streamName.length() == 0)
							continue;

						Playlist playlist = new Playlist(streamName);
						playlist.setRepeat((e.getAttribute("repeat").equals("false")) ? false : true);

						for (int j = 0; j < videos.getLength(); j++)
						{
							Node video = videos.item(j);
							if (video.getNodeType() == Node.ELEMENT_NODE)
							{
								Element e2 = (Element)video;
								String src = e2.getAttribute("src");
								Integer start = Integer.parseInt(e2.getAttribute("start"));
								Integer length = Integer.parseInt(e2.getAttribute("length"));
								playlist.addItem(src, start, length);
							}
						}
						String scheduled = e.getAttribute("scheduled");
						SimpleDateFormat parser = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
						Date startTime = null;
						try
						{

							startTime = parser.parse(scheduled);
						}
						catch (Exception z)
						{
							return "Parsing schedule time failed.";
						}
						Stream stream = streamsList.get(streamName);
						if (stream == null)
						{
							log.warn("ServerListenerStreamPublisher Stream does not exist for playlist: " + e.getAttribute("name") + " : " + streamName);
							continue;
						}

						List<ScheduledItem> schedules = schedulesMap.get(streamName);
						if (schedules == null)
						{
							schedules = new ArrayList<ScheduledItem>();
							schedulesMap.put(streamName, schedules);
						}
						stream.setSendOnMetadata(passMetaData);
						stream.setSwitchLog(switchLog);
						ScheduledItem schedule = new ScheduledItem(appInstance, startTime, playlist, stream);
						schedule.start();
						schedules.add(schedule);
						log.info("ServerListenerStreamPublisher Scheduled: " + stream.getName() + " for: " + scheduled);
					}
				}
			}
			catch (Exception ex)
			{
				return "Error from playlist manager is '" + ex.getMessage() + "'";
			}
			return "DONE!";
		}
	}

	public void shutdownStream(IApplicationInstance appInstance, Stream stream)
	{
		if (stream == null)
			return;

		synchronized(lock)
		{
			WMSProperties props = appInstance.getProperties();
			Map<String, List<ScheduledItem>> schedulesMap = (Map<String, List<ScheduledItem>>)props.get("streamPublisherSchedules");
			if(schedulesMap != null)
			{
				List<ScheduledItem> schedules = schedulesMap.remove(stream.getName());
				if(schedules != null)
				{
					for (ScheduledItem schedule : schedules)
					{
						schedule.stop();
					}
				}
			}
			props.remove(stream.getName());
			stream.closeAndWait();
			Publisher publisher = stream.getPublisher();
			if (publisher != null)
			{
				publisher.unpublish();
				publisher.close();
			}
			log.info("ServerListenerStreamPublisher: Stream shut down : " + stream.getName());
		}
	}

	public void onServerShutdownComplete(IServer server)
	{
	}

	public void onServerShutdownStart(IServer server)
	{
	}

	public void onServerConfigLoaded(IServer server)
	{
	}
}
