package com.wowza.wms.plugin.randomaccessreader.http;

import java.io.*;
import java.net.*;

import com.wowza.io.*;
import com.wowza.util.*;
import com.wowza.wms.logging.*;
import com.wowza.wms.stream.*;
import com.wowza.wms.application.*;
import com.wowza.wms.vhost.*;

public class HTTPRandomAccessReaderCached implements IRandomAccessReader
{
	private String basePath = "";
	private String mediaName = "";
	private String mediaExtension = "";
	private String fullPath = "";
	private File file = null;
	private RandomAccessFile randomFile = null;
	private int direction = IRandomAccessReader.FORWARD;
	
	private String id = "[unknown]";
	private URL url = null;
	private HTTPCacheItem cacheItem = null;
	
	public HTTPRandomAccessReaderCached()
	{
		WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).info("HTTPRandomAccessReaderCached.constructor");
	}
	
	public void init(IApplicationInstance appInstance, IMediaStream stream, String basePath, String mediaName, String mediaExtension)
	{
		this.basePath = basePath;
		this.mediaName = mediaName;
		this.mediaExtension = mediaExtension;
		
		WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).info("HTTPRandomAccessReaderCached.init: basePath:"+this.basePath+" mediaName:"+this.mediaName+" extension:"+this.mediaExtension);

		try
		{
			this.url = new URL(mediaName);
			this.id = URLUtils.urlToId(this.url);
			
			HTTPCacheSingleton cache = HTTPCacheSingleton.getInstance();
			synchronized(cache)
			{
				this.cacheItem = cache.get(this.id);
				if (this.cacheItem == null)
				{
					String cachePath = getCacheBasePath(VHostSingleton.getInstance(VHost.VHOST_DEFAULT));
					this.cacheItem = new HTTPCacheItem(this.url, this.id, cachePath);
					cache.put(this.id, this.cacheItem);
				}
			}
		}
		catch (Exception e)
		{
			WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).error("init: "+e.toString());
		}
		
		this.fullPath = this.basePath + File.separatorChar + this.id + "." + this.mediaExtension;
		this.file = this.cacheItem.getFile();
	}
	
	private String getCacheBasePath(IVHost vhost)
	{
		String homePath = vhost.getHomePath();
		homePath += "/cache";
		File cachePath = new File(homePath);
		if (!cachePath.exists())
			cachePath.mkdir();
		return cachePath.getPath();
	}

	public long getFilePointer()
	{
		if (this.randomFile == null)
			return 0;
		
		try
		{
			return this.randomFile.getFilePointer();
		}
		catch (Exception e)
		{
			WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).error("HTTPRandomAccessReaderCached.getFilePointer: "+e.toString());
		}
		return 0;
	}

	public void seek(long pos)
	{
		if (this.randomFile == null)
			return;
		
		try
		{
			this.cacheItem.blockForPos(pos);
			this.randomFile.seek(pos);
		}
		catch (Exception e)
		{
			WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).error("HTTPRandomAccessReaderCached.seek: "+e.toString());
		}
	}

	public int read(byte[] buf, int off, int size)
	{
		if (this.randomFile == null)
			return 0;
		
		try
		{
			long pos = this.randomFile.getFilePointer();
			this.cacheItem.blockForPos(pos+size);
			return this.randomFile.read(buf, off, size);
		}
		catch (Exception e)
		{
			WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).error("HTTPRandomAccessReaderCached.read: "+e.toString());
		}
		return 0;
	}

	public void open() throws IOException
	{
		WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).info("HTTPRandomAccessReaderCached.open["+this.mediaName+"]");
		if (this.randomFile != null)
			close();
		this.randomFile = new WowzaRandomAccessFile(this.file, "r");
	}

	public void close() throws IOException
	{
		WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).info("HTTPRandomAccessReaderCached.close["+this.mediaName+"]");
		if (this.randomFile != null)
			this.randomFile.close();
		this.randomFile = null;
	}

	public boolean isOpen()
	{
		return this.randomFile != null;
	}
	
	public int getDirecton()
	{
		return this.direction;
	}

	public void setDirecton(int direction)
	{
		this.direction = direction;
	}
			
	public String getBasePath()
	{
		return this.basePath;
	}

	public String getMediaName()
	{
		return this.mediaName;
	}

	public String getMediaExtension()
	{
		return mediaExtension;
	}
	
	public boolean exists()
	{
		return this.file.exists();
	}
	
	public long lastModified()
	{
		try
		{
			return this.file.lastModified();
		}
		catch (Exception e)
		{
			WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).error("HTTPRandomAccessReaderCached.lastModified: "+e.toString());
		}
		return 0;
	}
	
	public long length()
	{
		try
		{
			return this.cacheItem.getLength(); //this.file.length();
		}
		catch (Exception e)
		{
			WMSLoggerFactory.getLogger(HTTPRandomAccessReaderCached.class).error("HTTPRandomAccessReaderCached.length: "+e.toString());
		}
		return 0;
	}
	
	public String getPath()
	{
		return this.fullPath;
	}
}
